#Importing Libraries
import pandas as pd
import numpy as np
import ast
import matplotlib.pyplot as plt
from sklearn.linear_model import LinearRegression
from pygam import LinearGAM
from sklearn.preprocessing import MinMaxScaler
from sklearn.model_selection import train_test_split
from xgboost import XGBRegressor
from sklearn.model_selection import GridSearchCV
from sklearn.metrics import r2_score
import networkx as nx
from sklearn.model_selection import RandomizedSearchCV
from pygam import s
from sklearn.base import BaseEstimator, RegressorMixin

class GAMWrapper(BaseEstimator, RegressorMixin):
    def __init__(self, lam=0.6, n_splines=25, spline_order=3,max_iter=100):
        self.lam = lam
        self.n_splines = n_splines
        self.spline_order = spline_order
        self.max_iter = max_iter
        self.gam = LinearGAM(s(0, n_splines=self.n_splines, spline_order=self.spline_order), lam=self.lam)
    def fit(self, X, y):
        self.gam.fit(X, y)
        return self
    def predict(self, X):
        return self.gam.predict(X)





class SparseCausal():
    """
    SparseCausal Machine Learning (SparseCausal) class for causal inference.

    Parameters:
    - treatment_model: The model for estimating the treatment effect.
    - outcome_model: The model for estimating the outcome.
    - residual_model: The model for estimating the treatment effect residuals.
    """
    def __init__(self, treatment_model, outcome_model, residual_model):
        self.treatment_model = treatment_model
        self.outcome_model = outcome_model
        self.residual_model = residual_model
    def parameter_tuning(
    x_train: pd.DataFrame,
    y_train: pd.DataFrame,
    regressor,
    params: dict,
    n_iter: int,
):
        """
        Parameter tuning for xg_boost and random forest. Randomized search to get the best estimators for forecast.
        Args
            :param x_train: Training dataset with the regressors
            :param y_train: Target data
            :param x_test: Testing dataset with the regressors
            :param model_parameters: dictionary of parameters
            :param tuning: hyperparameter tuning
        Returns: 
            best parameters for model
        """
        random_search = RandomizedSearchCV(
            estimator = regressor,
            param_distributions = params,
            n_iter = n_iter,
            n_jobs = -1,
        )
        random_search.fit(x_train, y_train)
        return random_search

    def xg_boost(
        x_train: pd.DataFrame,
        y_train: pd.DataFrame,
        model_parameters: dict,
        tuning: bool = True,
    ):
        """
        xg_boost model - multivariate model with two options .True for hyperparameter tuning and False for fixed values for parameters.
        Input is a dataframe with target column and features created from create_features() function
        Args
            :param x_train: Training dataset with the regressors
            :param y_train: Target data
            :param x_test: Testing dataset with the regressors
            :param model_parameters: dictionary of parameters
            :param tuning: hyperparameter setting
        Returns: 
        model 
        """
        if tuning == True:          #True for hyperparameter tuning
            params = model_parameters['xg_boost']["True"]
            reg = SparseCausal.parameter_tuning(
                x_train = x_train,
                y_train = y_train,
                regressor = XGBRegressor(),
                params = params,
                n_iter = 10
            )
            m = reg.best_estimator_      #gives the best estimator
            m.fit(x_train,y_train)
            return m
        else:
            params = model_parameters['xg_boost']["False"]
            n_estimators = params["n_estimators"]
            learning_rate = params["learning_rate"]
            max_depth = params["max_depth"]
            n_jobs = params["n_jobs"]
            alpha = params["alpha"]
            reg = XGBRegressor(
                n_estimators = n_estimators,
                learning_rate = learning_rate,
                max_depth = max_depth,
                n_jobs = n_jobs,
                alpha = alpha
            )
            reg.fit(x_train,y_train)
            return (reg)
    def linear_gam(x_train,y_train,model_parameters,tuning):
        """
        Linear GAM model - multivariate model with two options .True for hyperparameter tuning and False for fixed values for parameters.
        Input is a dataframe with target column and features created from create_features() function
        Args
            :param x_train: Training dataset with the regressors
            :param y_train: Target data
            :param x_test: Testing dataset with the regressors
            :param model_parameters: dictionary of parameters
            :param tuning: hyperparameter setting
        Returns: 
            list containing the predicted values
        """
        if tuning == True:
            random_search = RandomizedSearchCV(GAMWrapper(), param_distributions=model_parameters['gam'], n_iter=50, cv=5, scoring='neg_mean_squared_error', random_state=42, n_jobs=-1)
            random_search.fit(x_train, y_train)

            reg = random_search.best_estimator_  
            reg.fit(x_train, y_train)
            return reg
        else:
            reg = LinearGAM(
            terms='auto'
        
            )
            reg.fit(x_train,y_train)
            return reg
    def lin_regression(x_train, y_train, model_parameters, tuning):
        """
        Perform linear regression on the provided training data.

        Parameters:
        - x_train (array-like or pd.DataFrame): The input features for training.
        - y_train (array-like or pd.Series): The target variable for training.
        - model_parameters: Any additional parameters for the linear regression model (not used in this simple example).
        - tuning: Any tuning parameters (not used in this simple example).

        Returns:
        - LinearRegression: The trained linear regression model.
        """
        if (tuning == True):
            random_search = SparseCausal.parameter_tuning(
                x_train = x_train,
                y_train = y_train,
                regressor = LinearRegression(),
                params = model_parameters['linear_regression']['True'],
                n_iter = 10
            )

            reg = random_search.best_estimator_  
            reg.fit(x_train, y_train)
            return reg


        else:
            # Create a Linear Regression model
            reg = LinearRegression()

            # Fit the model to the training data
            reg.fit(x_train, y_train)

            # Return the trained model
            return reg

            




    def treatment_model_selection(self, treatment_model, x_train, y_train, model_parameters, tuning):
        """
        Select and train the treatment model based on the specified type.

        Parameters:
        - treatment_model (str): The type of treatment model to be selected.
        - x_train (array-like or pd.DataFrame): The input features for training.
        - y_train (array-like or pd.Series): The target variable for training.
        - model_parameters: Any additional parameters for the treatment model.
        - tuning: Any tuning parameters for the treatment model.

        Returns:
        - object: The trained treatment model.
        """
        model_treatment = ''
        if treatment_model == "Linear GAM":
            model_treatment = SparseCausal.linear_gam(x_train, y_train, model_parameters, tuning)
        elif treatment_model == "xgboost":
            model_treatment = SparseCausal.xg_boost(x_train, y_train, model_parameters, tuning)
        elif treatment_model == "linear regression":
            model_treatment = SparseCausal.lin_regression(x_train, y_train, model_parameters, tuning)
        return model_treatment

    def outcome_model_selection(self, outcome_model, x_train, y_train, model_parameters, tuning):
        """
        Select and train the outcome model based on the specified type.

        Parameters:
        - outcome_model (str): The type of outcome model to be selected.
        - x_train (array-like or pd.DataFrame): The input features for training.
        - y_train (array-like or pd.Series): The target variable for training.
        - model_parameters: Any additional parameters for the outcome model.
        - tuning: Any tuning parameters for the outcome model.

        Returns:
        - object: The trained outcome model.
        """
        model_outcome = ''
        if outcome_model == "Linear GAM":
            model_outcome = SparseCausal.linear_gam(x_train, y_train, model_parameters, tuning)
        elif outcome_model == "xgboost":
            model_outcome = SparseCausal.xg_boost(x_train, y_train, model_parameters, tuning)
        elif outcome_model == "linear regression":
            model_outcome = SparseCausal.lin_regression(x_train, y_train, model_parameters, tuning)
        return model_outcome

    def residual_model_selection(self, residual_model, x_train, y_train, model_parameters, tuning):
        """
        Select and train the residual model based on the specified type.

        Parameters:
        - residual_model (str): The type of residual model to be selected.
        - x_train (array-like or pd.DataFrame): The input features for training.
        - y_train (array-like or pd.Series): The target variable for training.
        - model_parameters: Any additional parameters for the residual model.
        - tuning: Any tuning parameters for the residual model.

        Returns:
        - object: The trained residual model.
        """
        model_residual = ''
        if residual_model == "Linear GAM":
            model_residual = SparseCausal.linear_gam(x_train, y_train, model_parameters, tuning)
        elif residual_model == "xgboost":
            model_residual = SparseCausal.xg_boost(x_train, y_train, model_parameters, tuning)
        elif residual_model == "linear regression":
            model_residual = SparseCausal.lin_regression(x_train, y_train, model_parameters, tuning)
        return model_residual
    def select_best_correlated_pairs(Config,data, category):
        """
        Select the best correlated pairs of confounder and treatment variables 
        based on correlation coefficients.

        Parameters:
        - data (pd.DataFrame): The input DataFrame containing the data.
        - category (str): The category for which correlated pairs are selected.

        Returns:
        - pd.DataFrame: DataFrame containing the selected correlated pairs 
        of confounder and treatment variables.
        """

        # Calculate the correlation matrix
        correlation_matrix = data.drop('volume', axis=1).corr()

        # Reshape the correlation matrix for better visualization
        correlation_matrix = correlation_matrix.reset_index().melt(id_vars='index')
        correlation_matrix.columns = ['confounder variable', 'treatment variable', 'corr']

        # Exclude perfect correlations (corr=1.0) as they don't provide useful information , implicit definition for Directed Acyclic Graph
        correlation_matrix = correlation_matrix[correlation_matrix["corr"] != 1.0]

        # Sort the correlation matrix based on the absolute correlation coefficients
        selected_corr = correlation_matrix.sort_values(by=["corr"], key=abs, ascending=False)

        # Select the top correlated pairs based on the specified size
        selected_corr = selected_corr.head(Config.correlation_pair_size[category] * 2)

        # Extract only the relevant columns for the final result
        selected_corr = selected_corr[['confounder variable', 'treatment variable']]

        return selected_corr

    def sorted_alphabetic_order(x, y):
        """
        Combine two strings in alphabetic order separated by an underscore.

        Parameters:
        - x (str): The first string.
        - y (str): The second string.

        Returns:
        - str: The combined string in alphabetic order, separated by an underscore.
        """

        # Initialize an empty string to store the result
        result = ''

        # Check which string comes first in alphabetic order
        if x < y:
            result += x
            result += "_"
            result += y
        else:
            result += y
            result += "_"
            result += x

        return result
    
    def confounder_treatment_individual_treatment(Config,data, outcome_col, confounder_treatment_pair, category, channel):
        """
        Runs individsual treatment models for given confounder-treatment pairs and evaluates performance.

        Parameters:
        - data (DataFrame): Input data for analysis.
        - outcome_col (str): Column name for the outcome variable.
        - confounder_treatment_pair (DataFrame): DataFrame containing pairs of confounder and treatment variables.
        - category (str): The category for analysis.
        - channel (str): The channel for analysis.

        Returns:
        - DataFrame: Results dataframe with model performance metrics.
        """

        confounder_treatment_dataframe = pd.DataFrame(
            columns=['Category', 'Channel', 'Treatment model', 'Outcome model', 'Residual model', 'confounder variable',
                    'confounder_treatment_pair_key', 'treatment variable', 'Input type', 'R-squared_train', 'R-squared_test','ATE'])
        confounder_treatment_dataframe_sparse = pd.DataFrame(
            columns=['Category', 'Channel', 'Treatment model', 'Outcome model', 'Residual model', 'confounder variable',
                    'confounder_treatment_pair_key', 'treatment variable', 'Input type', 'R-squared_train',
                    'R-squared_test','ATE'])
        

        # Train-test split
        train_data, test_data = train_test_split(data, test_size=Config.test_size, random_state=Config.random_state)
        j = 1

        for model_treatment_type in Config.model_treatment_type_list:
            for model_outcome_type in Config.model_outcome_type_list:
                for model_residual_type in Config.model_residual_type_list:
                    print("\nIteration ", j)
                    print("Treatment Model: ", model_treatment_type)
                    print("Outcome Model: ", model_outcome_type)
                    print("Residual Model: ", model_residual_type)
                    j += 1
                    confounder_treatment_pair_copy = confounder_treatment_pair.copy()
                    confounder_treatment_pair_business = confounder_treatment_pair_copy[
                        confounder_treatment_pair_copy['Type of input'] == 'Business']
                    for i in range(len(confounder_treatment_pair_business)):
                        confounder_variable = confounder_treatment_pair_business.iloc[i]["confounder variable"]
                        treatment_variable = confounder_treatment_pair_business.iloc[i]["treatment variable"]
                        confounder_treatment = confounder_treatment_pair_business.iloc[i]["confounder-treatment-pair-key"]
                        input_type = confounder_treatment_pair_business.iloc[i]["Type of input"]
                        print(f"Pair {i + 1}: Features {confounder_variable} and {treatment_variable}")

                        # Select the correlated features data
                        feature_data = train_data[[confounder_variable, treatment_variable]].copy()

                        # Remove the treatment & outcome from the covariate features: assuming feature1 affects feature2
                        covariate_features = [f for f in train_data.columns if f not in [treatment_variable, outcome_col]]

                        # Step 1: Treatment model: predicts treatment variable using covariate features
                        X_treatment = train_data[covariate_features]

                        # Create a treatment model
                        model_treatment = SparseCausal(model_treatment_type, model_outcome_type,
                                                model_residual_type).treatment_model_selection(model_treatment_type,X_treatment,feature_data[treatment_variable],Config.model_parameters_dict,Config.tuning)

                        treatment_predict = model_treatment.predict(train_data[covariate_features])
                        treatment_predict_test = model_treatment.predict(test_data[covariate_features])

                        # Treatment residual using test dataset
                        treatment_residual = feature_data[treatment_variable] - treatment_predict
                        treatment_residual_test = test_data[treatment_variable] - treatment_predict_test

                        # Step 2: Outcome model: predict outcome variable using covariate features
                        model_outcome = SparseCausal(model_treatment_type, model_outcome_type,
                                                model_residual_type).treatment_model_selection(model_outcome_type,X_treatment,train_data[outcome_col],Config.model_parameters_dict,Config.tuning)

                        model_predict = model_outcome.predict(train_data[covariate_features])
                        model_predict_test = model_outcome.predict(test_data[covariate_features])

                        outcome_residual = train_data[outcome_col] - model_predict
                        outcome_residual_test = test_data[outcome_col] - model_predict_test

                        # Step 3: Residual model: predict outcome residuals using treatment residuals
                        model_residual = SparseCausal(model_treatment_type, model_outcome_type,
                                                model_residual_type).treatment_model_selection(model_residual_type,treatment_residual.values.reshape(-1, 1),outcome_residual.values,Config.model_parameters_dict,Config.tuning)

                        residual_predict = model_residual.predict(treatment_residual.values.reshape(-1, 1))
                        residual_predict_test = model_residual.predict(treatment_residual_test.values.reshape(-1, 1))

                        outcome_final_predict = model_predict + residual_predict
                        outcome_final_predict_test = model_predict_test + residual_predict_test

                        num = np.square(test_data[outcome_col].values - outcome_final_predict_test).sum()
                        den = np.square(
                            test_data[outcome_col].values - test_data[outcome_col].values.mean()).sum()
                        r_squared_test = 1 - (num / den)
                        r_squared = r2_score(train_data[outcome_col], outcome_final_predict)

                        print(f"R-squared train data {r_squared}")
                        print(f"R-squared test data  {r_squared_test}")
                        
                        # Calculate the ATE using the trestment residuals and outcome residuals
                        ate = np.mean(np.gradient(residual_predict_test,treatment_residual_test))

                        print("Average Treatment Effect (ATE) based on predictions:", ate)

                        confounder_treatment_dataframe = pd.concat([confounder_treatment_dataframe,pd.DataFrame([
                            {'Category': category, 'Channel': channel, 'Treatment model': model_treatment_type,
                            'Outcome model': model_outcome_type, 'Residual model': model_residual_type,
                            'confounder variable': confounder_variable, 'confounder_treatment_pair_key': confounder_treatment,
                            'treatment variable': treatment_variable, 'Input type': input_type,
                            'R-squared_train': r_squared, 'R-squared_test': r_squared_test,'ATE':ate}])], ignore_index=True)

                    confounder_treatment_pair_copy = confounder_treatment_pair.copy()
                    confounder_treatment_pair_sparse = confounder_treatment_pair_copy[
                        confounder_treatment_pair_copy['Type of input'] == 'Sparse']

                    for i in range(len(confounder_treatment_pair_sparse)):
                        confounder_variable = confounder_treatment_pair_sparse.iloc[i]["confounder variable"]
                        treatment_variable = confounder_treatment_pair_sparse.iloc[i]["treatment variable"]
                        confounder_treatment = confounder_treatment_pair_sparse.iloc[i]["confounder-treatment-pair-key"]
                        input_type = confounder_treatment_pair_sparse.iloc[i]["Type of input"]
                        print(f"Pair {i + 1}: Features {confounder_variable} and {treatment_variable}")

                        # Select the correlated features data
                        feature_data = train_data[[confounder_variable, treatment_variable]].copy()

                        # Remove the treatment & outcome from the covariate features: assuming feature1 affects feature2
                        covariate_features = [f for f in train_data.columns if
                                            f not in [treatment_variable, outcome_col]]

                        # Treatment model: predicts treatment variable using covariate features
                        X_treatment = train_data[covariate_features]

                        # Create a treatment model
                        model_treatment = SparseCausal(model_treatment_type, model_outcome_type,
                                                model_residual_type).treatment_model_selection(model_treatment_type,X_treatment,feature_data[treatment_variable], Config.model_parameters_dict, Config.tuning)

                        treatment_predict = model_treatment.predict(train_data[covariate_features])
                        treatment_predict_test = model_treatment.predict(test_data[covariate_features])

                        # Treatment residual using test dataset
                        treatment_residual = feature_data[treatment_variable] - treatment_predict
                        treatment_residual_test = test_data[treatment_variable] - treatment_predict_test

                        # Outcome model: predict outcome variable using covariate features
                        model_outcome = SparseCausal(model_treatment_type, model_outcome_type,
                                                model_residual_type).outcome_model_selection(model_outcome_type,X_treatment,train_data[outcome_col],Config.model_parameters_dict, Config.tuning)

                        model_predict = model_outcome.predict(train_data[covariate_features])
                        model_predict_test = model_outcome.predict(test_data[covariate_features])

                        outcome_residual = train_data[outcome_col] - model_predict
                        outcome_residual_test = test_data[outcome_col] - model_predict_test

                        # Residual model: predict outcome residuals using treatment residuals
                        model_residual = SparseCausal(model_treatment_type, model_outcome_type,
                                                model_residual_type).residual_model_selection(model_residual_type,treatment_residual.values.reshape(-1, 1), outcome_residual.values, Config.model_parameters_dict, Config.tuning)

                        residual_predict = model_residual.predict(treatment_residual.values.reshape(-1, 1))
                        residual_predict_test = model_residual.predict(treatment_residual_test.values.reshape(-1, 1))

                        outcome_final_predict = model_predict + residual_predict
                        outcome_final_predict_test = model_predict_test + residual_predict_test

                        num = np.square(test_data[outcome_col].values - outcome_final_predict_test).sum()
                        den = np.square(
                            test_data[outcome_col].values - test_data[outcome_col].values.mean()).sum()
                        r_squared_test = 1 - (num / den)
                        r_squared = r2_score(train_data[outcome_col], outcome_final_predict)

                        print(f"R-squared train data {r_squared}")
                        print(f"R-squared test data  {r_squared_test}")
                        # Calculate the ATE using the trestment residuals and outcome residuals
                        ate = np.mean(np.gradient(residual_predict_test,list(treatment_residual_test)))

                        print("Average Treatment Effect (ATE) based on predictions:", ate)
                        confounder_treatment_dataframe_sparse = pd.concat([confounder_treatment_dataframe_sparse,pd.DataFrame(
                            [{'Category': category, 'Channel': channel, 'Treatment model': model_treatment_type,
                            'Outcome model': model_outcome_type, 'Residual model': model_residual_type,
                            'confounder variable': confounder_variable,
                            'confounder_treatment_pair_key': confounder_treatment,
                            'treatment variable': treatment_variable, 'Input type': input_type,
                            'R-squared_train': r_squared, 'R-squared_test': r_squared_test,'ATE':ate}])], ignore_index=True)

        confounder_treatment_dataframe_sparse_indexes = confounder_treatment_dataframe_sparse.groupby(
            ['Category', 'Channel', 'Treatment model', 'Outcome model', 'Residual model',
            'confounder_treatment_pair_key'])['R-squared_test'].idxmax()
        
        # removing Bidirectionality only for Sparse casual potential pairs
        confounder_treatment_dataframe_sparse_bidirectional_removed = confounder_treatment_dataframe_sparse.loc[
            confounder_treatment_dataframe_sparse_indexes]
        
        confounder_treatment_dataframe_indexes = confounder_treatment_dataframe.groupby(
            ['Category', 'Channel', 'Treatment model', 'Outcome model', 'Residual model',
            'confounder_treatment_pair_key'])['R-squared_test'].idxmax()
        
        # removing Bidirectionality only for business casual potential pairs
        confounder_treatment_dataframe = confounder_treatment_dataframe.loc[
            confounder_treatment_dataframe_indexes]
        
        confounder_treatment_dataframe = pd.concat(
            [confounder_treatment_dataframe, confounder_treatment_dataframe_sparse_bidirectional_removed], axis=0)
        return confounder_treatment_dataframe
    def concurrent_integration_casual_model(Config,data, outcome_col, confounder_treatment_dataframe, confounder_treatment_pair, category, channel):
        """
        Runs concurrent integration of causal models considering with overall DAG

        Parameters:
        - data (DataFrame): Input data for analysis.
        - outcome_col (str): Column name for the outcome variable.
        - confounder_treatment_dataframe (DataFrame): DataFrame containing results of individual confounder-treatment models.
        - confounder_treatment_pair (DataFrame): DataFrame containing pairs of confounder and treatment variables.
        - category (str): The category for analysis.
        - channel (str): The channel for analysis.

        Returns:
        - DataFrame: Results dataframe with overall model performance metrics.
        """

        # Since we have multiple mutual causal dependencies in final DAG, we are calculating treatment effect of each
        # confounder-treatment pair, treating them simultaneously, and then calculating overall R2 score
        dag_type_len = Config.dag_type_min_len[category]
        Model_overall_result_dataframe = pd.DataFrame(
            ['Category', 'Channel', 'Treatment model', 'Outcome model', 'Residual model', 'Confounder-treatment pair',
            'covariate_features_list','treatment_features_list', 'Number of causal relationships', 'Train R2 score_overall', 'Test R2 score_overall'])

        # Train-test split
        train_data, test_data = train_test_split(data, test_size=Config.test_size, random_state=Config.random_state)
        
        i = 1
        for model_treatment_type in Config.model_treatment_type_list:
            for model_outcome_type in Config.model_outcome_type_list:
                for model_residual_type in Config.model_residual_type_list:
                    print("\nIteration ", i)
                    print("Treatment Model: ", model_treatment_type)
                    print("Outcome Model: ", model_outcome_type)
                    print("Residual Model: ", model_residual_type)
                    i += 1

                    confounder_treatment_pair_copy = confounder_treatment_dataframe.copy()
                    confounder_treatment_pair_filtered = confounder_treatment_pair_copy[
                        (confounder_treatment_pair_copy['Treatment model'] == model_treatment_type) & (
                                    confounder_treatment_pair_copy['Outcome model'] == model_outcome_type) & (
                                    confounder_treatment_pair_copy['Residual model'] == model_residual_type) &(confounder_treatment_pair_copy['Category'] == category) & (confounder_treatment_pair_copy['Channel'] == channel) ]

                    
                    dag_type_len = Config.dag_type_min_len[category]
                    while (dag_type_len <= len(confounder_treatment_pair_filtered)):
                        print("\nNumber of confounder-treatment considered in DAG", {dag_type_len - 1}, ":", dag_type_len)
                        confounder_treatment_pair_list = list(
                            zip(*map(confounder_treatment_pair_filtered.head(dag_type_len).get,
                                    ['confounder variable', 'treatment variable'])))
                        print("confounder-treatment pairs in DAG", {dag_type_len - 1}, ":", confounder_treatment_pair_list)
                        treatment_residual_array = []
                        treatment_residual_test_array = []
                        treatment_variable_array = []
                        treatment_outcome_array = []
                        treatment_variable_array = list(set(confounder_treatment_pair_filtered[
                            ["treatment variable"]].head(dag_type_len)['treatment variable'].tolist()))
                        treatment_outcome_array = treatment_variable_array.copy()
                        treatment_outcome_array.append(outcome_col)

                        covariate_features = [f for f in train_data.columns if f not in treatment_outcome_array]
                        print("Covariate Features Excluding treatment and Outcome Variables", covariate_features)
                        treat_features = [f for f in train_data.columns if f in treatment_variable_array]

                        # Treatment model
                        for i in range(0, len(treatment_variable_array), 1):
                            confounder_variable = confounder_treatment_pair_filtered.iloc[i]["confounder variable"]
                            treatment_variable = treatment_variable_array[i]
                            X_treatment = train_data[treatment_variable]
                            X_covariates = train_data[covariate_features]

                            model_treatment = SparseCausal(model_treatment_type, model_outcome_type, model_residual_type).treatment_model_selection(
                                model_treatment_type, X_covariates, train_data[treatment_variable], Config.model_parameters_dict,
                                Config.tuning)

                            treatment_predict = model_treatment.predict(X_covariates)

                            treatment_residual = X_treatment - treatment_predict
                            treatment_residual_array.append(treatment_residual)
                            treatment_predict_test = model_treatment.predict(test_data[covariate_features])
                            treatment_residual_test = test_data[treatment_variable] - treatment_predict_test
                            treatment_residual_test_array.append(treatment_residual_test)

                        # Outcome model
                        model_outcome = SparseCausal(model_treatment_type, model_outcome_type, model_residual_type).outcome_model_selection(
                            model_outcome_type, train_data[covariate_features], train_data[outcome_col],
                            Config.model_parameters_dict, Config.tuning)

                        model_predict = model_outcome.predict(train_data[covariate_features])
                        model_predict_test = model_outcome.predict(test_data[covariate_features])

                        outcome_residual = train_data[outcome_col] - model_predict

                        # Residual model: predict outcome residuals using treatment residuals
                        model_residual = SparseCausal(model_treatment_type, model_outcome_type, model_residual_type).residual_model_selection(
                            model_residual_type, np.transpose(treatment_residual_array), outcome_residual.values,
                            Config.model_parameters_dict, Config.tuning)

                        residual_predict = model_residual.predict(np.transpose(treatment_residual_array))

                        residual_predict_test = model_residual.predict(np.transpose(treatment_residual_test_array))

                        outcome_final_predict = model_predict + residual_predict
                        outcome_final_predict_test = model_predict_test + residual_predict_test

                        outcome_final_predict = model_predict + residual_predict

                        num = np.square(test_data[outcome_col].values - outcome_final_predict_test).sum()
                        den = np.square(test_data[outcome_col].values - test_data[outcome_col].values.mean()).sum()
                        r_squared_test = 1 - (num / den)
                        r_squared = r2_score(train_data[outcome_col], outcome_final_predict)

                        print(f"Overall R-squared train data {r_squared}")
                        print(f"Overall R-squared test data  {r_squared_test}")
                        Model_overall_result_dataframe = pd.concat([Model_overall_result_dataframe,pd.DataFrame(
                            [{'Category': category, 'Channel': channel, 'Treatment model': model_treatment_type,
                            'Outcome model': model_outcome_type, 'Residual model': model_residual_type,
                            'Confounder-treatment pair': confounder_treatment_pair_list,
                            'covariate_features_list': covariate_features,'treatment_features_list':treat_features,
                            'Number of causal relationships': dag_type_len, 'Train R2 score_overall': r_squared,
                            'Test R2 score_overall': r_squared_test}])], ignore_index=True)

                        dag_type_len += 1
        return Model_overall_result_dataframe
    def concurrent_integration_casual_model_sensitivity_treatment(Config,data, outcome_col, confounder_treatment_dataframe, confounder_treatment_pair, category, channel):
        """
        Runs concurrent integration of causal models considering multiple confounder-treatment pairs for sensitivity analysis considering the random shuffling of test data set for treatment variable.

        Parameters:
        - data (DataFrame): Input data for analysis.
        - outcome_col (str): Column name for the outcome variable.
        - confounder_treatment_dataframe (DataFrame): DataFrame containing results of individual confounder-treatment models.
        - confounder_treatment_pair (DataFrame): DataFrame containing pairs of confounder and treatment variables.
        - category (str): The category for analysis.
        - channel (str): The channel for analysis.

        Returns:
        - DataFrame: Results dataframe with overall model performance metrics.
        """

        # Since we have multiple mutual causal dependencies, we are calculating treatment effect of each
        # confounder-treatment pair, treating them simultaneously, and then calculating overall R2 score
        dag_type_len = Config.dag_type_min_len[category]
        Model_overall_result_dataframe = pd.DataFrame(
            ['Category', 'Channel', 'Treatment model', 'Outcome model', 'Residual model', 'Confounder-treatment pair',
            'covariate_features_list', 'Number of causal relationships', 'Train R2 score_overall', 'Test R2 score_overall_treatment_shuffle'])

        # Train-test split
        train_data, test_data = train_test_split(data, test_size=Config.test_size, random_state=Config.random_state)
        

        i = 1
        for model_treatment_type in Config.model_treatment_type_list:
            for model_outcome_type in Config.model_outcome_type_list:
                for model_residual_type in Config.model_residual_type_list:
                    print("\nIteration ", i)
                    print("Treatment Model: ", model_treatment_type)
                    print("Outcome Model: ", model_outcome_type)
                    print("Residual Model: ", model_residual_type)
                    i += 1

                    confounder_treatment_pair_copy = confounder_treatment_dataframe.copy()
                    confounder_treatment_pair_filtered = confounder_treatment_pair_copy[
                        (confounder_treatment_pair_copy['Treatment model'] == model_treatment_type) & (
                                    confounder_treatment_pair_copy['Outcome model'] == model_outcome_type) & (
                                    confounder_treatment_pair_copy['Residual model'] == model_residual_type)&(confounder_treatment_pair_copy['Category'] == category) & (confounder_treatment_pair_copy['Channel'] == channel)]

                    dag_type_len = Config.dag_type_min_len[category]
                    while (dag_type_len <= len(confounder_treatment_pair_filtered)):
                        print("\nNumber of confounder-treatment considered in DAG", {dag_type_len - 1}, ":", dag_type_len)
                        confounder_treatment_pair_list = list(
                            zip(*map(confounder_treatment_pair_filtered.head(dag_type_len).get,
                                    ['confounder variable', 'treatment variable'])))
                        print("confounder-treatment pairs in DAG", {dag_type_len - 1}, ":", confounder_treatment_pair_list)
                        treatment_residual_array = []
                        treatment_residual_test_array = []
                        treatment_variable_array = []
                        treatment_outcome_array = []
                        treatment_variable_array = list(set(confounder_treatment_pair_filtered[
                            ["treatment variable"]].head(dag_type_len)['treatment variable'].tolist()))
                        treatment_outcome_array = treatment_variable_array.copy()
                        treatment_outcome_array.append(outcome_col)

                        covariate_features = [f for f in train_data.columns if f not in treatment_outcome_array]
                        print("Covariate Features Excluding treatment and Outcome Variables", covariate_features)

                        # Treatment model
                        for i in range(0, len(treatment_variable_array), 1):
                            confounder_variable = confounder_treatment_pair_filtered.iloc[i]["confounder variable"]
                        
                            treatment_variable = treatment_variable_array[i]
                            X_treatment = train_data[treatment_variable]
                            X_covariates = train_data[covariate_features]

                            model_treatment = SparseCausal(model_treatment_type, model_outcome_type, model_residual_type).treatment_model_selection(
                                model_treatment_type, X_covariates, train_data[treatment_variable], Config.model_parameters_dict,
                                Config.tuning)

                            treatment_predict = model_treatment.predict(X_covariates)

                            treatment_residual = X_treatment - treatment_predict
                            treatment_residual_array.append(treatment_residual)
                            # Shuffle the test treatment variable for sensitivity analysis
                            test_data[treatment_variable]=np.random.permutation(test_data[treatment_variable].values)
                            treatment_predict_test = model_treatment.predict(test_data[covariate_features])
                            treatment_residual_test = test_data[treatment_variable] - treatment_predict_test
                            treatment_residual_test_array.append(treatment_residual_test)

                        # Outcome model
                        model_outcome = SparseCausal(model_treatment_type, model_outcome_type, model_residual_type).outcome_model_selection(
                            model_outcome_type, train_data[covariate_features], train_data[outcome_col],
                            Config.model_parameters_dict, Config.tuning)

                        model_predict = model_outcome.predict(train_data[covariate_features])
                        model_predict_test = model_outcome.predict(test_data[covariate_features])

                        outcome_residual = train_data[outcome_col] - model_predict

                        # Residual model: predict outcome residuals using treatment residuals
                        model_residual = SparseCausal(model_treatment_type, model_outcome_type, model_residual_type).residual_model_selection(
                            model_residual_type, np.transpose(treatment_residual_array), outcome_residual.values,
                            Config.model_parameters_dict, Config.tuning)

                        residual_predict = model_residual.predict(np.transpose(treatment_residual_array))

                        residual_predict_test = model_residual.predict(np.transpose(treatment_residual_test_array))

                        outcome_final_predict = model_predict + residual_predict
                        outcome_final_predict_test = model_predict_test + residual_predict_test

                        outcome_final_predict = model_predict + residual_predict

                        num = np.square(test_data[outcome_col].values - outcome_final_predict_test).sum()
                        den = np.square(test_data[outcome_col].values - test_data[outcome_col].values.mean()).sum()
                        r_squared_test = 1 - (num / den)
                        r_squared = r2_score(train_data[outcome_col], outcome_final_predict)

                        print(f"Overall R-squared train data {r_squared}")
                        print(f"Overall R-squared test data  {r_squared_test}")
                        Model_overall_result_dataframe = pd.concat([Model_overall_result_dataframe,pd.DataFrame(
                            [{'Category': category, 'Channel': channel, 'Treatment model': model_treatment_type,
                            'Outcome model': model_outcome_type, 'Residual model': model_residual_type,
                            'Confounder-treatment pair': confounder_treatment_pair_list,
                            'covariate_features_list': covariate_features,
                            'Number of causal relationships': dag_type_len, 'Train R2 score_overall': r_squared,
                            'Test R2 score_overall_treatment_shuffle': r_squared_test}])], ignore_index=True)

                        dag_type_len += 1
        return Model_overall_result_dataframe

    def concurrent_integration_casual_model_sensitivity_random_confounder(Config,data, outcome_col, confounder_treatment_dataframe, confounder_treatment_pair, category, channel):
        """
        Runs concurrent integration of causal models considering multiple confounder-treatment pairs for sensitivity analysis considering the random shuffling of test data set for treatment variable.

        Parameters:
        - data (DataFrame): Input data for analysis.
        - outcome_col (str): Column name for the outcome variable.
        - confounder_treatment_dataframe (DataFrame): DataFrame containing results of individual confounder-treatment models.
        - confounder_treatment_pair (DataFrame): DataFrame containing pairs of confounder and treatment variables.
        - category (str): The category for analysis.
        - channel (str): The channel for analysis.

        Returns:
        - DataFrame: Results dataframe with overall model performance metrics.
        """

        # Since we have multiple mutual causal dependencies, we are calculating treatment effect of each
        # confounder-treatment pair, treating them simultaneously, and then calculating overall R2 score
        dag_type_len = Config.dag_type_min_len[category]
        Model_overall_result_dataframe = pd.DataFrame(
            ['Category', 'Channel', 'Treatment model', 'Outcome model', 'Residual model', 'Confounder-treatment pair',
            'covariate_features_list', 'Number of causal relationships', 'Train R2 score_overall', 'Test R2 score_overall_random_confounder'])

        # Train-test split
        train_data, test_data = train_test_split(data, test_size=Config.test_size, random_state=Config.random_state)
        

        i = 1
        for model_treatment_type in Config.model_treatment_type_list:
            for model_outcome_type in Config.model_outcome_type_list:
                for model_residual_type in Config.model_residual_type_list:
                    print("\nIteration ", i)
                    print("Treatment Model: ", model_treatment_type)
                    print("Outcome Model: ", model_outcome_type)
                    print("Residual Model: ", model_residual_type)
                    i += 1

                    confounder_treatment_pair_copy = confounder_treatment_dataframe.copy()
                    confounder_treatment_pair_filtered = confounder_treatment_pair_copy[
                        (confounder_treatment_pair_copy['Treatment model'] == model_treatment_type) & (
                                    confounder_treatment_pair_copy['Outcome model'] == model_outcome_type) & (
                                    confounder_treatment_pair_copy['Residual model'] == model_residual_type)&(confounder_treatment_pair_copy['Category'] == category) & (confounder_treatment_pair_copy['Channel'] == channel)]


                    dag_type_len = Config.dag_type_min_len[category]

                    while (dag_type_len <= len(confounder_treatment_pair_filtered)):
                        print("\nNumber of confounder-treatment considered in DAG", {dag_type_len - 1}, ":", dag_type_len)
                        confounder_treatment_pair_list = list(
                            zip(*map(confounder_treatment_pair_filtered.head(dag_type_len).get,
                                    ['confounder variable', 'treatment variable'])))
                        print("confounder-treatment pairs in DAG", {dag_type_len - 1}, ":", confounder_treatment_pair_list)
                        treatment_residual_array = []
                        treatment_residual_test_array = []
                        treatment_variable_array = []
                        treatment_outcome_array = []
                        treatment_variable_array = list(set(confounder_treatment_pair_filtered[
                            ["treatment variable"]].head(dag_type_len)['treatment variable'].tolist()))
                        treatment_outcome_array = treatment_variable_array.copy()
                        treatment_outcome_array.append(outcome_col)

                        covariate_features = [f for f in train_data.columns if f not in treatment_outcome_array]
                        print("Covariate Features Excluding treatment and Outcome Variables", covariate_features)

                        # Treatment model
                        for i in range(0, len(treatment_variable_array), 1):
                            confounder_variable = confounder_treatment_pair_filtered.iloc[i]["confounder variable"]
                            treatment_variable = treatment_variable_array[i]
                            X_treatment = train_data[treatment_variable]
                            
                            X_covariates = train_data[covariate_features]
                            

                            model_treatment = SparseCausal(model_treatment_type, model_outcome_type, model_residual_type).treatment_model_selection(
                                model_treatment_type, X_covariates, train_data[treatment_variable], Config.model_parameters_dict,
                                Config.tuning)

                            treatment_predict = model_treatment.predict(X_covariates)

                            treatment_residual = X_treatment - treatment_predict
                            treatment_residual_array.append(treatment_residual)
                            
                            # Shuffle the confounder variable for sensitivity analysis
                            test_data[confounder_variable]=np.random.permutation(test_data[confounder_variable].values)
                            treatment_predict_test = model_treatment.predict(test_data[covariate_features])
                            treatment_residual_test = test_data[treatment_variable] - treatment_predict_test
                            treatment_residual_test_array.append(treatment_residual_test)

                        # Outcome model
                        model_outcome = SparseCausal(model_treatment_type, model_outcome_type, model_residual_type).outcome_model_selection(
                            model_outcome_type, train_data[covariate_features], train_data[outcome_col],
                            Config.model_parameters_dict, Config.tuning)

                        model_predict = model_outcome.predict(train_data[covariate_features])
                        model_predict_test = model_outcome.predict(test_data[covariate_features])

                        outcome_residual = train_data[outcome_col] - model_predict

                        # Residual model: predict outcome residuals using treatment residuals
                        model_residual = SparseCausal(model_treatment_type, model_outcome_type, model_residual_type).outcome_model_selection(
                            model_residual_type, np.transpose(treatment_residual_array), outcome_residual.values,
                            Config.model_parameters_dict, Config.tuning)

                        residual_predict = model_residual.predict(np.transpose(treatment_residual_array))

                        residual_predict_test = model_residual.predict(np.transpose(treatment_residual_test_array))

                        outcome_final_predict = model_predict + residual_predict
                        outcome_final_predict_test = model_predict_test + residual_predict_test

                        outcome_final_predict = model_predict + residual_predict

                        num = np.square(test_data[outcome_col].values - outcome_final_predict_test).sum()
                        den = np.square(test_data[outcome_col].values - test_data[outcome_col].values.mean()).sum()
                        r_squared_test = 1 - (num / den)
                        r_squared = r2_score(train_data[outcome_col], outcome_final_predict)

                        print(f"Overall R-squared train data {r_squared}")
                        print(f"Overall R-squared test data  {r_squared_test}")
                        Model_overall_result_dataframe = pd.concat([Model_overall_result_dataframe,pd.DataFrame(
                            [{'Category': category, 'Channel': channel, 'Treatment model': model_treatment_type,
                            'Outcome model': model_outcome_type, 'Residual model': model_residual_type,
                            'Confounder-treatment pair': confounder_treatment_pair_list,
                            'covariate_features_list': covariate_features,
                            'Number of causal relationships': dag_type_len, 'Train R2 score_overall': r_squared,
                            'Test R2 score_overall_random_confounder': r_squared_test}])], ignore_index=True)

                        dag_type_len += 1
        return Model_overall_result_dataframe
    def best_model_selection(Config,Model_overall_result_dataframe_all, Model_overall_result_dataframe_all_sensitivity_random_confounder, Model_overall_result_dataframe_all_sensitivity_treatment):
        """
        Selects the best model based on a weighted combination of test R2 scores and sensitivity scores.

        Parameters:
        - Model_overall_result_dataframe_all: DataFrame containing overall model results
        - Model_overall_result_dataframe_all_sensitivity_random_confounder: DataFrame with sensitivity results for random confounder
        - Model_overall_result_dataframe_all_sensitivity_treatment: DataFrame with sensitivity results for treatment shuffle

        Returns:
        - DataFrame with the best model selected for each category and channel
        """
        if((Config.iterations_type=="all") & (Config.type_of_model=="Business")):
            dag_len_dataframe = pd.DataFrame.from_dict(Config.dag_type_len,orient ='index')
            dag_len_dataframe.reset_index(inplace=True)
            dag_len_dataframe.columns = ['Category','min number of causal relationships']
            Model_overall_result_dataframe_all = Model_overall_result_dataframe_all.merge(dag_len_dataframe,on=['Category'],how='inner')
            Model_overall_result_dataframe_all = Model_overall_result_dataframe_all[Model_overall_result_dataframe_all['Number of causal relationships']==Model_overall_result_dataframe_all['min number of causal relationships']]
        elif((Config.iterations_type=="all")&(Config.type_of_model=="Blended")):
            dag_len_dataframe = pd.DataFrame.from_dict(Config.dag_type_len,orient ='index')
            dag_len_dataframe.reset_index(inplace=True)
            dag_len_dataframe.columns = ['Category','min number of causal relationships']
            Model_overall_result_dataframe_all = Model_overall_result_dataframe_all.merge(dag_len_dataframe,on=['Category'],how='inner')
            Model_overall_result_dataframe_all = Model_overall_result_dataframe_all[Model_overall_result_dataframe_all['Number of causal relationships']>=Model_overall_result_dataframe_all['min number of causal relationships']]
        elif(Config.iterations_type=="individual_combinations"):
            Model_overall_result_dataframe_all=Model_overall_result_dataframe_all


        # Extract relevant columns for sensitivity analysis
        Model_overall_result_dataframe_all_sensitivity_random_confounder = Model_overall_result_dataframe_all_sensitivity_random_confounder[['Category', 'Channel', 'Treatment model', 'Outcome model',
        'Residual model', 'Confounder-treatment pair',
        'covariate_features_list', 'Number of causal relationships', 'Test R2 score_overall_random_confounder']]

        Model_overall_result_dataframe_all_sensitivity_treatment = Model_overall_result_dataframe_all_sensitivity_treatment[['Category', 'Channel', 'Treatment model', 'Outcome model',
            'Residual model', 'Confounder-treatment pair',
            'covariate_features_list', 'Number of causal relationships', 'Test R2 score_overall_treatment_shuffle']]

        ## Best model selection 
        # Merge dataframes for overall results and sensitivity analysis
        best_model_data_frame = Model_overall_result_dataframe_all.merge(Model_overall_result_dataframe_all_sensitivity_random_confounder, on=['Category', 'Channel', 'Treatment model', 'Outcome model',
            'Residual model', 'Confounder-treatment pair',
            'covariate_features_list', 'Number of causal relationships'], how='left')
        best_model_data_frame = best_model_data_frame.merge(Model_overall_result_dataframe_all_sensitivity_treatment, on=['Category', 'Channel', 'Treatment model', 'Outcome model',
            'Residual model', 'Confounder-treatment pair',
            'covariate_features_list', 'Number of causal relationships'], how='left')

        # Fill NaN values with 0
        best_model_data_frame[['Test R2 score_overall_treatment_shuffle', 'Test R2 score_overall_random_confounder']] = best_model_data_frame[['Test R2 score_overall_treatment_shuffle', 'Test R2 score_overall_random_confounder']].fillna(value=0)

        # Calculate the overall model score based on weighted combination of test R2 scores
        best_model_data_frame['Model_score'] = (Config.weight_test_r2 * best_model_data_frame['Test R2 score_overall'] +
                                                Config.weight_sensitivity_test_treatment_r2 * best_model_data_frame['Test R2 score_overall_treatment_shuffle'] +
                                                Config.weight_sensitivity_test_random_confounder_r2 * best_model_data_frame['Test R2 score_overall_random_confounder'])

        # Identify the index of the best model based on the highest model score for each category and channel
        best_model_data_frame_indexes = best_model_data_frame.groupby(['Category', 'Channel'])['Model_score'].idxmax()

        # Select the rows corresponding to the best models
        best_model_data_frame_final = best_model_data_frame.loc[best_model_data_frame_indexes]

        # Identify the index of the best model based on the business inputs alone for each category and channel
        best_model_data_frame_business_alone = best_model_data_frame.copy()
        best_model_data_frame_business_alone = best_model_data_frame_business_alone[(best_model_data_frame_business_alone['Number of causal relationships']==best_model_data_frame_business_alone['min number of causal relationships'])]
        best_model_data_frame_business_indexes = best_model_data_frame_business_alone.groupby(['Category', 'Channel'])['Model_score'].idxmax()

        # Select the rows corresponding to the best models
        best_model_data_frame_final_business = best_model_data_frame_business_alone.loc[best_model_data_frame_business_indexes]

        return best_model_data_frame_final,best_model_data_frame_final_business
    def draw_dag(Config,relations):

        relations2 = relations.explode('covariate_features_list')
        unique_kpi = []
        unique_kpi.extend(relations['confounder variable'].tolist())
        unique_kpi.extend(relations['treatment variable'].tolist())
        unique_kpi.extend(relations2['covariate_features_list'].tolist())
        unique_kpi = set(unique_kpi)
        

        # Define the causal graph
        G = nx.DiGraph()
        
        # Adding outcome node
        G.add_node(Config.outcome_col, color='lightcoral')

        # All KPIs to outcome arrows & nodes
        for kpi in unique_kpi:
            G.add_node(kpi, color='orange')
            G.add_edge(kpi, Config.outcome_col, color='darkgreen')
        
        # Confounders to treatment arrows
        for index, relation in relations.iterrows():
            if(relation['Input type']=='Sparse'):
                G.add_edge(relation['confounder variable'], relation['treatment variable'], color='orange',label='Sparse')
            elif(relation['Input type']=='Business'):
                G.add_edge(relation['confounder variable'], relation['treatment variable'], color='blue',label='Business')

            
        # Visualize the DAG
        pos = nx.spring_layout(G)
        edges = G.edges()
        colors = [G[u][v]['color'] for u, v in edges]
        labels = {node: node for node in G.nodes()}

        nx.draw(G, pos, with_labels=True, labels=labels, font_weight='bold',edge_color=colors, arrowsize=20)
        plt.show()
    def best_model_selection_objects(Config,data, outcome_col, best_model_data_frame, category, channel):
        """
        Selects the best causal model based on the optimal combination of confounder-treatment pairs and model types.

        Parameters:
        - data (DataFrame): The dataset used for analysis.
        - outcome_col (str): The name of the column representing the outcome variable.
        - best_model_data_frame (DataFrame): A DataFrame containing results from individual confounder-treatment models.
        - category (str): The specific category to analyze.
        - channel (str): The channel associated with the analysis.

        Returns:
        - tuple: A tuple containing treatment model objects, residual model, outcome model, attribution DataFrame,
                average secondary_variables attribution percentage, and the baseline outcome model.
        """

        train_data = data
        secondary_variables_attribution_dataframe =pd.DataFrame()
        
        model_treatment_type = best_model_data_frame['Treatment model'].head(1).to_list()[0]
        model_outcome_type = best_model_data_frame['Outcome model'].head(1).to_list()[0]
        model_residual_type = best_model_data_frame['Residual model'].head(1).to_list()[0]

        data_string = best_model_data_frame["treatment_features_list"].iloc[0]
        print(data_string)
        # Convert the string representation of the list to a list
        treatment_features = list(ast.literal_eval(data_string))


        treatment_residual_array = []
        treatment_residual_test_array = []
        treatment_variable_array = []
        treatment_outcome_array = []
        treatment_outcome_array = treatment_features.copy()
        treatment_outcome_array.append(outcome_col)

        covariate_features = [f for f in train_data.columns if f not in treatment_outcome_array]
        print("Covariate Features Excluding treatment and Outcome Variables", covariate_features)

        # Baseline prediction
        model_outcome_baseline = SparseCausal(model_treatment_type, model_outcome_type, model_residual_type).outcome_model_selection(model_outcome_type, train_data[covariate_features], train_data[outcome_col],
            Config.model_parameters_dict, Config.tuning)

        # Baseline estimation
        input_data_baseline = pd.DataFrame(train_data[covariate_features]).copy()
        for col in covariate_features:
            if(col in Config.baseline_primary_feature):
                input_data_baseline[col] = 0
            else:
                input_data_baseline[col] = np.median(input_data_baseline[col])
        
        baseline = np.array(model_outcome_baseline.predict(input_data_baseline))
        

        secondary_variables_attribution_dataframe = pd.DataFrame()
        secondary_variables_attribution_dataframe[Config.outcome_col]=train_data[Config.outcome_col]

        model_baseline_predict = model_outcome_baseline.predict(train_data[covariate_features])
        
        baseline = (baseline)*train_data[outcome_col]/(baseline+model_baseline_predict)
        

        secondary_variables_attribution_dataframe['Baseline'] = pd.Series(baseline)

        model_treatment_dict = {}
        covariate_features = [col for col in covariate_features if col not in Config.baseline_features]

        for i in range(0, len(treatment_features), 1):
            treatment_variable = treatment_features[i]

            X_treatment = train_data[treatment_variable]
            X_covariates = train_data[covariate_features]

            model_treatment = SparseCausal(model_treatment_type, model_outcome_type, model_residual_type).treatment_model_selection(
                model_treatment_type, X_covariates, train_data[treatment_variable], Config.model_parameters_dict,
                Config.tuning)

            treatment_predict = model_treatment.predict(X_covariates)

            treatment_residual = X_treatment - treatment_predict
            treatment_residual_array.append(treatment_residual)
    
            model_treatment_dict[treatment_variable]= model_treatment
        

        model_outcome = SparseCausal(model_treatment_type, model_outcome_type, model_residual_type).outcome_model_selection(model_outcome_type, train_data[covariate_features], train_data[outcome_col]-baseline,Config.model_parameters_dict, Config.tuning)
        
        model_predict = model_outcome.predict(train_data[covariate_features])

        outcome_residual = train_data[outcome_col] - model_predict

        # # Subtracting outcome residual from baseline
        
        model_residual = SparseCausal(model_treatment_type, model_outcome_type, model_residual_type).residual_model_selection(model_residual_type, np.transpose(treatment_residual_array), outcome_residual.values, Config.model_parameters_dict, Config.tuning)
        
        residual_predict = model_residual.predict(np.transpose(treatment_residual_array))
        secondary_variables_attribution_dataframe.reset_index(inplace=True,drop=True)
        secondary_variables_attribution_dataframe['residual_predict']= pd.Series(residual_predict)
        
        
        secondary_variables_attribution_dataframe['secondary_variables_attribution'] = pd.Series((train_data[outcome_col]-baseline).values-np.array(residual_predict)*(train_data[outcome_col]-baseline).values/train_data[outcome_col].values)

        division = secondary_variables_attribution_dataframe['secondary_variables_attribution'].values/np.array(baseline+model_predict)

        secondary_variables_attribution_dataframe['secondary_variables_attribution%'] = np.where(secondary_variables_attribution_dataframe[Config.outcome_col]!=0,division,0)

        

        avg_secondary_variables_attrition_perc = secondary_variables_attribution_dataframe['secondary_variables_attribution%'].mean()

    
        overall_kpi_list = covariate_features.copy()
        
        for treatment_feature in treatment_features:
            overall_kpi_list.append(treatment_feature)
        
        print("----Overall KPI List-------\n", overall_kpi_list)
        overall_kpi_list = [kpi for kpi in overall_kpi_list if kpi not in Config.baseline_features]
        for kpi in overall_kpi_list:
            kpi_att=pd.DataFrame()
            kpi_att[outcome_col] = train_data[outcome_col]

            for col in overall_kpi_list:
                if(col==kpi):
                    kpi_att[col] = train_data[col]
                else:
                    kpi_att[col] = np.median(train_data[col])
                    
            treatment_residual_kpi_arr =[]
            for i in range(0, len(treatment_features), 1):
                treatment_variable = treatment_features[i]

                X_treatment = kpi_att[treatment_variable]
                X_covariates = kpi_att[covariate_features]

                treatment_predict_kpi = model_treatment.predict(X_covariates)

                treatment_residual_kpi = X_treatment - treatment_predict_kpi
                treatment_residual_kpi_arr.append(treatment_residual_kpi)
        
        
        
            model_predict = model_outcome.predict(kpi_att[covariate_features])
            outcome_residual = kpi_att[outcome_col] - model_predict

            
            residual_predict_kpi = model_residual.predict(np.transpose(treatment_residual_kpi_arr))
            

            
            model_predict_kpi = model_outcome.predict(kpi_att[covariate_features])

            outcome_residual = kpi_att[outcome_col] - model_predict_kpi

            secondary_variables_attribution_dataframe[kpi+'_secondary_variables_attribution'] = pd.Series((train_data[outcome_col]-baseline).values-np.array(residual_predict_kpi)*(train_data[outcome_col]-baseline).values/train_data[outcome_col].values)
            
        
        secondary_variables_attribution_dataframe['Category'] = category
        secondary_variables_attribution_dataframe['Channel'] = channel
        
        att_list =[]
        for kpi in overall_kpi_list:
            att_list.append(kpi+'_secondary_variables_attribution')
        secondary_variables_attribution_dataframe['kpi_attribution_sum']=secondary_variables_attribution_dataframe[att_list]. sum(axis=1)

        for kpicol in overall_kpi_list:
            secondary_variables_attribution_dataframe[kpicol+'_secondary_variables_attribution'] = np.divide(secondary_variables_attribution_dataframe[kpicol+'_secondary_variables_attribution'].values*secondary_variables_attribution_dataframe['secondary_variables_attribution'].values,
            (secondary_variables_attribution_dataframe['kpi_attribution_sum'].values))

            secondary_variables_attribution_dataframe[kpicol+'_secondary_variables_attribution%'] = np.divide(secondary_variables_attribution_dataframe[kpicol+'_secondary_variables_attribution'].values*secondary_variables_attribution_dataframe['secondary_variables_attribution%'].values,secondary_variables_attribution_dataframe['secondary_variables_attribution'].values)
        secondary_variables_attribution_dataframe['kpi_attribution_sum']=secondary_variables_attribution_dataframe[att_list]. sum(axis=1)

        return model_treatment_dict, model_residual, model_outcome,secondary_variables_attribution_dataframe,avg_secondary_variables_attrition_perc,model_outcome_baseline
    def category_channel_data_prep(Config,data, category, channel):
        """
        Prepares data for a specific category and channel.

        Parameters:
        - data (DataFrame): The input DataFrame with wide-format data.
        - category (str): The desired category for analysis.
        - channel (str): The desired channel for analysis.

        Returns:
        - DataFrame: Processed data for the specified category and channel.
        """

        # Selecting Channel and Category
        channel_level = data[(data[Config.granularity1] == channel) & (data[Config.granularity2] == category)]

        # Dropping columns that have all null values
        channel_level = channel_level.dropna(how='all', axis=1)

        # Filling any missing (NaN) values with 0
        channel_level = channel_level.fillna(0)
        
        # Define the columns to be used for mean and sum aggregations
        column_to_be_used = channel_level.columns[7:]

        # Group by 'channel', 'global_category', and 'survey_week' and apply aggregation functions
        data = channel_level.groupby(Config.groupby_columns_input).agg(
            {
                Config.groupby_agg_column1: 'mean',
                Config.groupby_agg_column2: 'mean',
                **{col: 'mean' for col in column_to_be_used}
            }
        ).reset_index()

        # Dropping unnecessary columns
        data.drop(Config.drop_columns_input, axis=1, inplace=True)

        # Defining our Outcome and features column
        outcome_col = Config.outcome_col
        selected_features = data.columns

        # Create a MinMaxScaler instance
        scaler = MinMaxScaler()

        # Select the data to be scaled, in this case, the 'outcome_col' column
        data_to_scale = data[[outcome_col]]

        # Fit the scaler to the data, which computes the minimum and maximum values
        scaler.fit(data_to_scale)

        # Transform (scale) the data using the fitted scaler
        scaled_data = scaler.transform(data_to_scale)

        # Update the 'outcome_col' column with the scaled values
        data[outcome_col] = scaled_data

        # Now, the 'outcome_col' column contains values scaled to the range [0, 1]
        return data,scaler

    def SparseCausalDriver(Config,wide_df):
        """
        Executes a causal inference pipeline on specified categories and channels using a sparse causal model.

        This function performs the following tasks:
        1. Prepares the data for each category and channel.
        2. Identifies the top correlated confounder-treatment pairs through correlation analysis.
        3. Incorporates business feedback into the confounder-treatment pairs.
        4. Runs individual treatment models for each category and channel.
        5. Conducts concurrent integration of causal models.
        6. Performs sensitivity analyses by shuffling treatments and introducing random confounders.
        7. Selects the best models based on R-squared scores.
        8. Stores the final model objects and attributions for secondary variables.

        Parameters:
        -----------
        Config : object
            Configuration object that contains the outcome column, channel list, category list, feedbacks, eliminations, 
            type of model, and exception cuts. It also includes methods and settings required for data preparation and 
            model execution.
        wide_df : pandas.DataFrame
            The input wide-format DataFrame containing the data for all channels and categories.

        Returns:
        --------
        tuple
            Returns a tuple containing the following elements:
            - confounder_treatment_dataframe_all : pandas.DataFrame
                DataFrame containing all confounder-treatment pairs across all categories and channels.
            - Model_overall_result_dataframe_all : pandas.DataFrame
                DataFrame containing overall results of the causal models, including R-squared scores.
            - Model_overall_result_dataframe_all_sensitivity_treatment : pandas.DataFrame
                DataFrame containing results of the sensitivity analysis with treatment shuffling.
            - Model_overall_result_dataframe_all_sensitivity_random_confounder : pandas.DataFrame
                DataFrame containing results of the sensitivity analysis with random confounder introduction.
            - best_model_data_frame_final : pandas.DataFrame
                DataFrame containing the best selected models for each category and channel based on the overall results.
            - best_model_data_frame_final_business : pandas.DataFrame
                DataFrame containing the best selected models based on business criteria.
            - secondary_variables_attribution_dataframe_all : pandas.DataFrame
                DataFrame containing the attribution percentages for secondary variables across all categories and channels.
        """
        # Define the outcome column using the configuration
        outcome_col = Config.outcome_col
        print(outcome_col)

        # Initialize the overall result dataframe
        Model_overall_result_dataframe_all = pd.DataFrame(
            columns=['Category', 'Channel', 'Treatment model', 'Outcome model', 'Residual model', 'Confounder-treatment pair',
                    'covariate_features_list','treatment_features_list', 'Number of causal relationships', 'Train R2 score_overall', 'Test R2 score_overall'])

        # Initialize the sensitivity analysis dataframe for treatment shuffling
        Model_overall_result_dataframe_all_sensitivity_treatment = pd.DataFrame(
            columns=['Category', 'Channel', 'Treatment model', 'Outcome model', 'Residual model', 'Confounder-treatment pair',
                    'covariate_features_list', 'Number of causal relationships', 'Train R2 score_overall', 'Test R2 score_overall_treatment_shuffle'])

        # Initialize the sensitivity analysis dataframe for random confounder introduction
        Model_overall_result_dataframe_all_sensitivity_random_confounder = pd.DataFrame(
            columns=['Category', 'Channel', 'Treatment model', 'Outcome model', 'Residual model', 'Confounder-treatment pair',
                    'covariate_features_list', 'Number of causal relationships', 'Train R2 score_overall', 'Test R2 score_overall_random_confounder'])

        confounder_treatment_dataframe_all = pd.DataFrame(
            columns=['Category', 'Channel', 'Treatment model', 'Outcome model', 'Residual model', 'confounder variable',
                    'confounder_treatment_pair_key', 'treatment variable', 'Input type', 'R-squared_train', 'R-squared_test','ATE'])
        scaler_dict_all={}

        final_model_objects_all ={}
        final_model_objects={}
        secondary_variables_attribution_dataframe_all = pd.DataFrame()
        # Iterate over channels and categories
        for channel in Config.channel_list:
            for category in Config.category_list:
                if (channel, category) not in Config.exception_cut:
                    print((channel, category))
                    
                    # Prepare data for the specified category and channel
                    data,scaler = SparseCausal.category_channel_data_prep(Config,wide_df, category, channel)

                    # Pairwise correlation is a foundational step - potential indicator for causation,
                    # can be spurious as well which we will decide based on business call
                    top_correlated_pairs = SparseCausal.select_best_correlated_pairs(Config,data, category)
                    top_correlated_pairs['Type of input'] = 'Sparse'
                    top_correlated_pairs['confounder-treatment-pair-key'] = top_correlated_pairs.apply(
                        lambda x: SparseCausal.sorted_alphabetic_order(x['confounder variable'], x['treatment variable']), axis=1)

                    # Define feedbacks for the category
                    feedbacks = Config.feedbacks[category]

                    # Create a dataframe with confounder-treatment pairs based on feedbacks
                    confounder_treatment_pair = pd.DataFrame(feedbacks, columns=['confounder variable', 'treatment variable'])
                    confounder_treatment_pair['Type of input'] = 'Business'
                    print(confounder_treatment_pair)
                    confounder_treatment_pair['confounder-treatment-pair-key'] = confounder_treatment_pair.apply(
                        lambda x: SparseCausal.sorted_alphabetic_order(x['confounder variable'], x['treatment variable']), axis=1)

                    # For common causal pairs, we are giving precedence to Business Feedback on the right directions
                    confounder_treatment_pair_common_cuts = confounder_treatment_pair.merge(
                        top_correlated_pairs,
                        on=['confounder-treatment-pair-key', 'confounder variable', 'treatment variable'], how='inner')
                    confounder_treatment_pair_common_cuts['Type of input'] = np.where(
                        confounder_treatment_pair_common_cuts["Type of input_x"].isnull() & (~confounder_treatment_pair_common_cuts["Type of input_y"].isnull()),
                        confounder_treatment_pair_common_cuts["Type of input_y"],
                        np.where(confounder_treatment_pair_common_cuts["Type of input_y"].isnull() & (~confounder_treatment_pair_common_cuts["Type of input_x"].isnull()),confounder_treatment_pair_common_cuts["Type of input_x"],"Business & Sparse"))
                    print(confounder_treatment_pair_common_cuts)
                    common_cuts = list(confounder_treatment_pair_common_cuts['confounder-treatment-pair-key'].unique())
                    print(common_cuts)

                    # For missing intersections from business, potential causal pairs are considered from the Sparse approach
                    confounder_treatment_pair_missing_cuts = top_correlated_pairs[
                        ~(top_correlated_pairs['confounder-treatment-pair-key'].isin(common_cuts))]

                    # Overall confounder treatment pairs
                    if(Config.type_of_model=="Blended"):
                        confounder_treatment_pair = pd.concat(
                            [confounder_treatment_pair, confounder_treatment_pair_missing_cuts], axis=0)
                    elif(Config.type_of_model=="Business"):
                        confounder_treatment_pair=confounder_treatment_pair

                    print(confounder_treatment_pair.columns)
                    
                    # Define eliminunabated feedbacks for category
                    if(category=="Chocolate"):
                        elimination_feedbacks = Config.elimination[category]

                        # Create a dataframe with confounder-treatment pairs based on feedbacks
                        elimination_confounder_treatment_pair = pd.DataFrame(elimination_feedbacks, columns=['confounder variable', 'treatment variable'])

                        elimination_confounder_treatment_pair['confounder-treatment-pair-key'] = elimination_confounder_treatment_pair.apply(
                            lambda x: SparseCausal.sorted_alphabetic_order(x['confounder variable'], x['treatment variable']), axis=1)
                        
                        # Elimination confounder-treatment-pair-key
                        elimination_confounder_treatment_pair_list = list(elimination_confounder_treatment_pair['confounder-treatment-pair-key'].unique())
                    
                        # Create a dataframe with confounder-treatment pairs based on updated feedback after elimination
                        confounder_treatment_pair =confounder_treatment_pair[~confounder_treatment_pair['confounder-treatment-pair-key'].isin( elimination_confounder_treatment_pair_list)]
                    # Run individual treatment models for the given category and channel
                    confounder_treatment_dataframe = SparseCausal.confounder_treatment_individual_treatment(Config,data, outcome_col,
                                                                                                confounder_treatment_pair,
                                                                                                category, channel)

                    # Run concurrent integration of causal models for the given category and channel
                    Model_overall_result_dataframe = SparseCausal.concurrent_integration_casual_model(Config,data, outcome_col,
                                                                                        confounder_treatment_dataframe,
                                                                                        confounder_treatment_pair,
                                                                                        category, channel)
                    # Run Sensitivity analysis and get the ranomized treatment test data set R2 score of causal models for the given category and channel
                    Model_overall_result_dataframe_sensitivity_treatment = SparseCausal.concurrent_integration_casual_model_sensitivity_treatment(Config,data, outcome_col,
                                                                                        confounder_treatment_dataframe,
                                                                                        confounder_treatment_pair,
                                                                                        category, channel)
                    # Run Sensitivity analysis and get the ranomized confounder introduction test data set R2 score of causal models for the given category and channel
                    Model_overall_result_dataframe_sensitivity_random_confounder = SparseCausal.concurrent_integration_casual_model_sensitivity_random_confounder(Config,data, outcome_col,
                                                                                        confounder_treatment_dataframe,
                                                                                        confounder_treatment_pair,
                                                                                        category, channel)

                    # Concatenate the results to the overall result dataframe
                    Model_overall_result_dataframe_all = pd.concat(
                        [Model_overall_result_dataframe_all, Model_overall_result_dataframe], axis=0, ignore_index=True)
                    
                    # Concatenate the results to the overall result dataframe for sensitivity of treatment randomisation
                    Model_overall_result_dataframe_all_sensitivity_treatment = pd.concat(
                        [Model_overall_result_dataframe_all_sensitivity_treatment, Model_overall_result_dataframe_sensitivity_treatment], axis=0, ignore_index=True)
                    
                    # Concatenate the results to the overall result dataframe for sensitivity of random confounder introduction
                    Model_overall_result_dataframe_all_sensitivity_random_confounder = pd.concat(
                        [Model_overall_result_dataframe_all_sensitivity_random_confounder, Model_overall_result_dataframe_sensitivity_random_confounder], axis=0, ignore_index=True)
                    
                    # Concatenate the results to the overall result dataframe for sensitivity of treatment randomisation
                    confounder_treatment_dataframe_all = pd.concat(
                        [confounder_treatment_dataframe_all, confounder_treatment_dataframe], axis=0, ignore_index=True)
                    
                    # Scaler model addition 
                    scaler_dict_all[category]={channel:scaler}
                    
        # Convert certain columns to string type
        Model_overall_result_dataframe_all['Confounder-treatment pair'] = Model_overall_result_dataframe_all[
            'Confounder-treatment pair'].astype(str)
        Model_overall_result_dataframe_all['covariate_features_list'] = Model_overall_result_dataframe_all[
            'covariate_features_list'].astype(str)
        Model_overall_result_dataframe_all['treatment_features_list'] = Model_overall_result_dataframe_all[
            'treatment_features_list'].astype(str)

        # Drop unnecessary columns
        Model_overall_result_dataframe_all.drop(columns=[0], inplace=True)
        Model_overall_result_dataframe_all.dropna(subset=['covariate_features_list', 'Treatment model'], inplace=True)

        # Convert certain columns to string type
        Model_overall_result_dataframe_all_sensitivity_treatment['Confounder-treatment pair'] = Model_overall_result_dataframe_all_sensitivity_treatment[
            'Confounder-treatment pair'].astype(str)
        Model_overall_result_dataframe_all_sensitivity_treatment['covariate_features_list'] = Model_overall_result_dataframe_all_sensitivity_treatment[
            'covariate_features_list'].astype(str)

        # Drop unnecessary columns

        Model_overall_result_dataframe_all_sensitivity_treatment.dropna(subset=['covariate_features_list', 'Treatment model'], inplace=True)

        # Convert certain columns to string type
        Model_overall_result_dataframe_all_sensitivity_random_confounder['Confounder-treatment pair'] = Model_overall_result_dataframe_all_sensitivity_random_confounder[
            'Confounder-treatment pair'].astype(str)
        Model_overall_result_dataframe_all_sensitivity_random_confounder['covariate_features_list'] = Model_overall_result_dataframe_all_sensitivity_random_confounder[
            'covariate_features_list'].astype(str)

        # Drop unnecessary columns

        Model_overall_result_dataframe_all_sensitivity_random_confounder.dropna(subset=['covariate_features_list', 'Treatment model'], inplace=True)

        # Drop unnecessary columns
        confounder_treatment_dataframe_all.dropna(subset=['Treatment model'], inplace=True)

        best_model_data_frame_final,best_model_data_frame_final_business=SparseCausal.best_model_selection(Config,Model_overall_result_dataframe_all, Model_overall_result_dataframe_all_sensitivity_random_confounder, Model_overall_result_dataframe_all_sensitivity_treatment)



        # Iterate over channels and categories
        for channel in Config.channel_list:
            for category in Config.category_list:
                # Get the best model objects for current cut
                best_model_data_frame = best_model_data_frame_final[(best_model_data_frame_final['Channel']==channel) & (best_model_data_frame_final['Category']==category)]
                model_treatment_type = best_model_data_frame['Treatment model'].head(1).to_list()[0]
                model_outcome_type = best_model_data_frame['Outcome model'].head(1).to_list()[0]
                model_residual_type = best_model_data_frame['Residual model'].head(1).to_list()[0]
                # Prepare data for the specified category and channel
                data,scaler = SparseCausal.category_channel_data_prep(Config,wide_df, category, channel)
                outcome_col = Config.outcome_col
                model_treatment_dict,model_residual,model_outcome,secondary_variables_attribution_dataframe,avg_secondary_variables_attrition_perc,model_outcome_baseline = SparseCausal.best_model_selection_objects(Config,data, outcome_col, best_model_data_frame, category, channel)

                # Storing final trained objects, R2 scores in dictionaries
                final_model_objects_all[ category, channel]={"treatment":model_treatment_dict,"outcome":model_outcome,"residual":model_residual,"baseline":model_outcome_baseline,"ps_attribution_percentage":avg_secondary_variables_attrition_perc}
                
                secondary_variables_attribution_dataframe['Channel'] = channel
                secondary_variables_attribution_dataframe['Category'] = category
                secondary_variables_attribution_dataframe_all = pd.concat([secondary_variables_attribution_dataframe_all,secondary_variables_attribution_dataframe])

        return confounder_treatment_dataframe_all,Model_overall_result_dataframe_all,Model_overall_result_dataframe_all_sensitivity_treatment,Model_overall_result_dataframe_all_sensitivity_random_confounder,best_model_data_frame_final,best_model_data_frame_final_business,secondary_variables_attribution_dataframe_all


